#include <stdint.h>
#include <signal.h>
#include <msp430x20x3.h>

#define Bitime  0x45            // For UART transmission at 115.2Kbaud/sec
#define TXD_1	0x02		// TXD on P1.1

unsigned int TXData;
unsigned char BitCnt;

void SetupClock(void)
{
  BCSCTL1 = CALBC1_8MHZ;                      // uses internal sub-master clock at 8MHz
  DCOCTL = CALDCO_8MHZ;                       // derives the digitally-controlled oscillator from the internal submaster clock, again at 8MHz
}


void SetupIO(void)
{
   P2SEL &= 0x00;                               //I/O    
   
   TI_CC_GDO0_PxDIR &= 0x00;                    //input
   TI_CC_GDO0_PxIES = TI_CC_GDO0_PIN;           //int on falling edge (end of pkt)
   TI_CC_GDO0_PxIFG = 0;                        //clear flag; Port1.3
   TI_CC_GDO0_PxIE = TI_CC_GDO0_PIN;            //enable int on end of packet  

   P2SEL &= ~0x40;
   P2DIR &= ~0x40;   
}

void SetupUART (void)
{
	TACCTL0 = OUT;			// Timer A Capture/Compare Control 0 Register: Output bit = sets the output high. TXD Idle as Mark
	TACTL = TASSEL_2 + MC_1;	// Timer A Control Register: clock source = SMCLK, up mode - counts up to TACCR0 (see below)
	TACCTL0 = CM0 + CCIE;		// Timer A Capture/Compare Control 0 Register: Capture Mode = 'no capture', and enable interrupts
						// at the request of the corresponding CCIFG flag (capture compare interrupt flag)
						// This will be changed to 'capture mode' later on when we actually transmit our frame.
	TACCR0 = Bitime;			// Timer A Capture/Compare Register 0: We're using 'compare mode' meaning that the value Bitime is
						// compared to the actual timer (TAR - Timer A Register). Basically it's the value that our timer (TAR)
						// counts up to, determined by comparing its current value with the value Bittime.
	P1SEL  |= TXD_1;			// Port 1 Select Register: Here we're choosing the special function for port 1 pin 5, that is p1.5. In this
						// case the special function is Timer A_0
	P1DIR  |= TXD_1;			// And as such we choose this to be an output
}

void TX_Byte (void)
{

        BitCnt = 0xA;			// Load Bit counter, 8data + ST/SP (10 bits in total = A)
	TXData |= 0x100;			// Add mark stop bit to TXData
	TXData = TXData << 1;				// Add space start bit - remember start bit is 0, and stop bit is 1.
	TACCTL0 =  CM1 + CCIS0 + OUTMOD0 + CCIE;	// Timer A Capture/Compare Control 0 Register: set to capture on rising edge, capture an internal event on CCISA, enable interrupts 
								// TXD = mark = idle 
	while ( CCTL0 & CCIE );				// Wait for TX completion
}


int main (void)
{  
    WDTCTL = WDTPW + WDTHOLD;                     //stop WDT

   SetupClock();
   SetupIO();
   SetupUART();
   
   _EINT();				        // Enable interrupts   
   _BIS_SR(GIE);                                //enable interrupts  
   
   for(;;)
   {
     
     TxData = 0xAA;
     TX_Byte();
     TxData = 0xEE;
     TX_Byte();
   }
}


#pragma vector = TIMERA0_VECTOR
__interrupt void TIMERA0_ISR (void)
{
	// TX Routine
	if ( BitCnt == 0)
		TACCTL0 &= ~CCIE;			// All bits transmitted, disable interrupt
	else
	{
		TACCTL0 |=  OUTMOD2;		// TX Space - toggle the output mode

		if (TXData & 0x01)		// if least significant bit of TXData is high, then set the output bit high correspondingly??
			TACCTL0 &= ~ OUTMOD2;	// TX Mark - reset the output mode??
		TXData = TXData >> 1;		// shift the bit to the right by one bit (onto the next bit)
		BitCnt --;				// update the bit counter
	}
}

